<?php

namespace App\Models;

use EloquentFilter\Filterable;
use Illuminate\Database\Eloquent\Model;

use Laravel\Scout\Searchable;
use Cviebrock\EloquentSluggable\Sluggable;
use Illuminate\Database\Eloquent\SoftDeletes;

class Products extends Model
{
    use Filterable,Searchable,SoftDeletes,Sluggable;

    protected $dates=['deleted_at'];

    public function scopeActiveProducts($query)
    {
        return $query->where('status', 1);
    }

    public function sluggable()
    {
        return [
            'slug' => [
                'source' => 'name'
            ]
        ];
    }

    protected $table = 'products';

    protected $fillable = [
        'code', 'name', 'eng_name', 'slug', 'type', 'stuffType', 'brand', 'category',
        'sub_category', 'stuffType_id', 'brand_id', 'category_id', 'sub_category_id',
        'qty', 'short_description', 'base_img_url','base_medium_url', 'base_thumbnail_url',
        'base_sell_price', 'discount_sell_price', 'alarm_level',
        'review', 'weight', 'view_count', 'sold_count', 'status', 'available','buy'
    ];

    protected $casts = [
        'tec_info_keyAndValues' => 'array',
    ];


    /**
     * Get the indexable data array for the model.
     *
     * @return array
     */
    public function toSearchableArray()
    {
        return [
            'id'            => $this->id,
            'name'          => $this->name,
            'eng_name'      => $this->eng_name,
            'brand'         => $this->brand,
            'category'      => $this->category,
            'sub_category'  => $this->sub_category,
            'stuffType'     => $this->stuffType,
        ];
    }

    public function UserFavorite(){
        $this->belongsToMany(User::class,'Favorites','product_id','user_id');
    }

    public function Brand()
    {
        return $this->belongsTo(Brands::class, 'brand_id');
    }

    public function Category()
    {
        return $this->belongsTo(Category::class, 'category_id');
    }

    public function StuffType()
    {
        return $this->belongsTo(StuffType::class, 'stuffType_id');
    }

    public function productsGallery()
    {
        return $this->hasMany(ProductsGallery::class, 'product_id');
    }

    public function productsColor()
    {
        return $this->hasMany(ProductsColor::class, 'product_id');
    }

    public function productsRating()
    {
        return $this->hasMany(ProductsRating::class, 'product_id');
    }

    public function Review()
    {
        return $this->hasMany(Review::class, 'product_id');
    }

    public function getTecInfo(){

        return $this->hasMany(TecInfoProduct::class,'product_id');
    }

    public function getProductId(){
        $id = 'PRD-';
        $id .= sprintf("%02d",$this->StuffType->id);
        $id .= sprintf("%02d",$this->Category->id);
        $id .= sprintf("%02d",$this->id);

        return $id;
    }

    public function path()
    {
        return "/product/".$this->slug;
    }

    public function dashboardPath()
    {
        return "/CRM/Anbar/product/".$this->id;
    }

    public function checkAvailable(){
        $productsColor = $this->productsColor()
            ->where('status','!=',0)
            ->where('available','!=',0)->get();
        if(($productsColor->isEmpty())){
            return false;
        }
        if($this->status != 0){
            if($this->qty != 0 And $this->available != 0){
                return true;
            }
        }

        return false;
    }

    public function getImage(){
        if(!is_null($this->base_thumbnail_url)){
            return '<img src="'.url('/').$this->base_thumbnail_url.'" class="img-responsive front" alt="'.$this->name.' - '.$this->eng_name.'" title="'.$this->name.' - '.$this->eng_name.'">';
        }
        return '<img src="'.path_to_images('defaults/default.png').'"  class="img-responsive front"  style="opacity: 0.6" alt="'.$this->name.' - '.$this->eng_name.'" title="'.$this->name.' - '.$this->eng_name.'" >';
    }

    public function getImageCat(){
        if(!is_null($this->base_thumbnail_url)){
            return '<img src="'.url('/').$this->base_thumbnail_url.'" class="img-responsive front product-thumb-image" alt="'.$this->name.' - '.$this->eng_name.'" title="'.$this->name.' - '.$this->eng_name.'">';
        }
        return '<img src="'.path_to_images('defaults/default.png').'"  class="img-responsive front product-thumb-image"  style="opacity: 0.6" alt="'.$this->name.' - '.$this->eng_name.'" title="'.$this->name.' - '.$this->eng_name.'" >';
    }

    public function getPrice(){
        $html = '';

        if($this->checkAvailable()){
            if(!is_null($this->discount_sell_price)){
                $html .= '<span class="price_sale"><span
                        class="money"><span>تومان</span>'.priceFormat($this->discount_sell_price).'</span></span>
            <del class="price_compare">
                <span class="money">'.priceFormat($this->base_sell_price).'</span>
            </del>';
            }else{
                $html .= '<span class="price_sale"><span
                        class="money"><span>تومان</span>'.priceFormat($this->base_sell_price).'</span></span>';
            }
        }else{
            $html .= '<p>تماس بگیرید</p>';
        }

        return $html;
    }

    public function getPriceCompare()
    {
        $html = '';

        if ($this->checkAvailable()) {

            $html .= '<div class="price-box">
                                                    <span>' . priceFormat($this->base_sell_price) . '</span>
                                                    <span>تومان</span>
                                                </div>';

        } else {
            $html .= '<div class="price-box">
                                             
                                                    <span>ناموجود</span>
                                                </div>';
        }

        return $html;
    }

    public function getPercent(){

        $percentage = discount($this->base_sell_price,$this->discount_sell_price,$this);

        $html = '';
        if($percentage !== false){
            $html .= '<span class="sale_banner">
                            <span class="sale_text">-'.$percentage.'%</span>
                      </span>';
        }
//        return $html;
        return '';

    }

    public function getCart()
    {
        $html = '';
        if ($this->checkAvailable()){

            $html .= '<form class="pricing_forms add-to-cart-form" action = "'.route('addToCart2').'"
                  method = "POST"
                  enctype = "multipart/form-data" >
                
                    
                        '.csrf_field().'
                
                <input type = "hidden"
                       name = "productId"
                       value = "'.$this->id.'" >
        
                <button class="add-to-cart-btn"
                        type = "submit" >
                    <i class="fa fa-cart-plus sabad" ></i >
                </button >
        
            </form >';
        }

        return $html;

    }

    public function getRate(){

        $totalRate = Review::totalRate($this->id);

        if(!is_null($totalRate))
            return  '<span>'.$totalRate.'</span>';
        else
            return  '<span>بدون امتیاز</span>';

    }

    public function scopeActiveAvailableProducts($query)
    {
        return $query->where('status', 1)->where('available',1)->where('qty','!=',0);
    }

    public  function scopeAvailableFirst($query){
        return $query->orderBy('available','desc')->orderBy('qty','desc');
    }

    public function pdfs(){
        return $this->hasMany(ProductPdf::class,'product_id');
    }
}
