<?php

namespace App\Http\Controllers\CrmSiteFront\orders;

use App\Events\CartSmsEvent;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;;
use App\Models\Cart;
use App\Models\CartProduct;
use App\Models\ProductsColor;
use App\Models\User;
use Illuminate\Support\Facades\Event;

class CartsController extends Controller
{

    protected function indexPending(){

        $carts = Cart::getUnConfirmedCarts();


        return view('CRM.SiteFront.carts.pending',compact('carts'));
    }

    protected function indexConfirmed(){

        $carts = Cart::getConfirmedCarts();

        return view('CRM.SiteFront.carts.confirmed',compact('carts'));
    }

    protected function indexUserCarts(User $user){
        $carts_products = $user->Cart()
            ->where('paid',1)
            ->join('cart_products', 'carts.id', '=', 'cart_products.cart_id')
            ->select('cart_products.id','cart_products.cart_id','cart_products.product_color_id','cart_products.qty','cart_products.discount_sell_price','carts.created_at')
            ->latest()->paginate(16);

        return view('CRM.SiteFront.carts.users-cart',compact('carts_products','user'));

    }

    protected function indexUsers(){

        $users = User::where('confirmed',1)->get();

        return view('CRM.SiteFront.carts.users-list',compact('users'));
    }

    protected function createCart(User $user){

        $cart = $user->createCart();
        $cart_products = $cart->CartProduct()->latest()->get();
        $productColors = ProductsColor::where('qty','!=',0)->latest()->get();

        return view('CRM.SiteFront.carts.edit',compact('cart','cart_products','productColors'));
    }

    protected function editCart(Cart $cart){

        $cart_products = $cart->CartProduct()->latest()->get();
        $productColors = ProductsColor::where('qty','!=',0)->latest()->get();

        return view('CRM.SiteFront.carts.edit',compact('cart','cart_products','productColors'));
    }

    protected function updateCartAdmin(Request $request,Cart $cart){

        $this->validate($request,[
            'productColorId'    => 'required|numeric|min:1|exists:products_color,id',
            'qty'               => 'required|numeric|min:1'
        ]);
        $request = $request->all();

        $oldCartProduct = $cart->CartProduct()->where('product_color_id',$request['productColorId'])->first();

        if(isset($oldCartProduct)){
            alert()->flash('انجام نشد', 'info', [
                'text'              => "این کالا در این سبد خرید وجود دارد؛ اگر نیاز به تغییر تعداد کالا دارید، ابتدا حذف کرده و دوباره اضافه کنید",
                'showConfirmButton' => true,
            ]);

            return redirect()->back();
        }

        $productColor   = ProductsColor::whereId($request['productColorId'])->first();
        $cartProduct    = $this->findCartProduct($cart,$productColor);
        $status         = $this->checkInventory($productColor);

        //800 error is custom error(it means we dont have that product with that color any more)
        if($status == 800){

            $this->removeFromCart($cartProduct);

            alert()->flash('کالا موجود نمی باشد', 'error', [
                'text'              => "موجودی این کالا در انبار صفر است",
                'showConfirmButton' => true,
            ]);

            return redirect()->back();
        }

        $status = $this->checkInventoryLimit($productColor,$cartProduct,$request['qty']);

        //700 error is custom error(it means the requested qty is more than our inventories amount)
        if($status == 700){

            alert()->flash('بیش از حد مجاز', 'error', [
                'text'              => "تعداد کالای درخواستی بیشتر از موجود انبار است",
                'showConfirmButton' => true,
            ]);

            return redirect()->back();
        }

        CartProduct::create([
            'cart_id'                   => $cart->id,
            'product_color_id'          => $request['productColorId'],
            'qty'                       => $request['qty'],
            'base_sell_price'           => $productColor->base_sell_price,
            'discount_sell_price'       => $productColor->discount_sell_price,
            'discount'                  => $productColor->discount,
            'total'                     => $productColor->discount_sell_price * $request['qty'],
        ]);

        alert()->flash('اضافه شد', 'success', [
            'text' => "کالا به سبد خرید اضافه شد",
            'timer' => 1200,
        ]);


        return redirect()->back();

    }

    protected function deleteCartProduct(CartProduct $cartProduct){

        $cart = $cartProduct->Cart()->first();

        if($cart->paid == 0){
            deleteRecord($cartProduct,'کالا با موفقیت از سبد خرید حذف شد');
        }else{
            alert()->flash('انجام نشد', 'warning', [
                'text' => "کالاها در سبد خرید پرداخت شده قابل حذف نمی باشد",
                'showConfirmButton' => true,
            ]);
        }

        return redirect()->back();
    }

    protected function confirmCart(Cart $cart){

        $cart->confirmed = 1;
        $cart->save();

        Event::fire(new CartSmsEvent($cart));

        alert()->flash('تایید شد', 'success', [
            'text' => "سبد خرید با موفقیت تایید شد",
            'showConfirmButton' => false,
            'timer' => 1200
        ]);

        return redirect()->back();

    }

    protected function pendingCart(Cart $cart){

        if($cart->paid == 1){
            alert()->flash('توجه', 'warning', [
                'text' => "این سبد خرید پرداخت شده و قابلیت تغییر وضعیت ندارد",
                'showConfirmButton' => true,
            ]);
            return redirect()->back();
        }
        $cart->confirmed = 0;
        $cart->save();

        alert()->flash('تعلیق شد', 'success', [
            'text' => "سبد خرید با موفقیت تعلیق شد",
            'showConfirmButton' => false,
            'timer' => 1200
        ]);

        return redirect()->back();

    }

    protected function checkInventory(ProductsColor $productsColor){

        $product = $productsColor->Product()->first();

        if(isset($productsColor) && $product->qty != 0 && $productsColor->qty != 0){
            return 100;
        }

        return 800;
    }

    //800 error is custom error(it means we dont have that product with that color any more)
    //700 error is custom error(it means the requested qty is more than our inventories qty)
    //check if we have those qty of product or not
    public function checkInventoryLimit(ProductsColor $productsColor,$cartProduct,$qty){

        if(!is_null($cartProduct)){
            $qty = $cartProduct->qty + $qty;
        }

        if($qty > $productsColor->qty || $productsColor->qty == 0){
            $status = 700;
        }else{
            $status = 100;
        }

        return $status;
    }

    protected function findCartProduct(Cart $cart,ProductsColor $productsColor){
        return  $cartItem = $cart->CartProduct()->where('product_color_id',$productsColor->id)->first();
    }

    protected function removeFromCart($cartProduct){
        if(isset($cartProduct)){
            try{
                $cartProduct->forceDelete();
            }catch(\Illuminate\Database\QueryException $e){

            }
        }
    }
}
