<?php

namespace App\Http\Controllers\Auth;

use App\Events\RegisterEvent;
use App\Http\Requests\StepTwoRequest;
use App\Models\Cart;
use App\Models\User;
use App\Http\Controllers\Controller;
use App\Models\UserActivationSms;
use Carbon\Carbon;
use Illuminate\Auth\Events\Registered;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Event;
use Illuminate\Support\Facades\URL;
use Illuminate\Support\Facades\Validator;
use Illuminate\Foundation\Auth\RegistersUsers;
use Illuminate\Http\Request;

class RegisterController extends Controller
{
    /*
    |--------------------------------------------------------------------------
    | Register Controller
    |--------------------------------------------------------------------------
    |
    | This controller handles the registration of new users as well as their
    | validation and creation. By default this controller uses a trait to
    | provide this functionality without requiring any additional code.
    |
    */

    use RegistersUsers;

    /**
     * Create a new controller instance.
     *
     */
    public function __construct()
    {
        $this->middleware('guest');
        $this->middleware('prevent-back-history');
    }

    /**
     * Show the application registration form.
     *
     * @return \Illuminate\Http\Response
     */
    public function showRegistrationForm()
    {
        if (Auth::check()) {
            return redirect()->home();
        } else {
            return view('/public/SiteFront/auth/registerStep1');
        }
    }

    /**
     * Handle a registration request for the application.
     *
     * @param  \Illuminate\Http\Request $request
     * @return \Illuminate\Http\Response
     */
    public function register(Request $request)
    {
        if ($user=User::whereMobile($request->mobile)->first()){

            if ($user->confirmed == 0){
                alert()->flash('شما قبلا ثبت نام کرده اید.', 'warning', [
                    'text' => " شماره موبایل را تایید کنید.",
                    'showConfirmButton' => false,
                    'timer' => 2000
                ]);

                return redirect(route('register.activation'));
            }

            alert()->flash('شما قبلا ثبت نام کرده اید.', 'warning', [
                'text' => " وارد شوید.",
                'showConfirmButton' => false,
                'timer' => 2000
            ]);
            return redirect(\url('/login'));
        }


        $this->validator($request->all())->validate();

        event(new Registered($user = $this->create($request->all())));

        return $this->registered($request, $user) ?: redirect($this->redirectPath($user));
    }

    /**
     * Get a validator for an incoming registration request.
     *
     * @param  array $data
     * @return \Illuminate\Contracts\Validation\Validator
     */
    protected function validator(array $data)
    {
        return Validator::make($data, [
            'mobile' => ['required', 'digits:11', 'regex:/^(0)?(9[0-9]{9})$/'],
            'password' => 'required|string|min:6|max:60',
        ],[
            'mobile.unique'=>'شما قبلا ثبت نام کرده اید. لطفا از قسمت ورود وارد سایت شوید.'
        ]);

    }

    /**
     * Create a new user instance after a valid registration.
     *
     * @param  array $data
     * @return \App\Models\User
     */
    protected function create(array $data)
    {
        $token = encrypt_decrypt('encrypt', $data['mobile']);

    

        $user = User::create([
            'mobile' => $data['mobile'],
            'token' => $token,
            'password' => bcrypt($data['password'])
        ]);

        Event::fire(new RegisterEvent($user));

        return $user;
    }

    protected function redirectTo($user)
    {

        $token = encrypt_decrypt('encrypt', $user->mobile);

        return '/activation/?refurl=' . $token;
    }

    public function redirectPath($user)
    {
        if (method_exists($this, 'redirectTo')) {
            return $this->redirectTo($user);
        }

        return property_exists($this, 'redirectTo') ? $this->redirectTo : '/';
    }

    protected function storeStepTwo(StepTwoRequest $request)
    {

        $request = $request->all();
        $user = User::whereId(Auth::id())->firstOrFail();
        $user->update([
            'fname'             => $request['fname'],
            'lname'             => $request['lname'],
            'melli_code'        => $request['melli_code'],
            'tell'              => $request['tell'],
            'urgent_phone_no'   => $request['urgent_phone_no'],
            'email'             => $request['email'],
            'birth'             => $request['year'] . '/' . $request['month'] . '/' . $request['date'],
            'gender'            => $request['gender'],
            'state'             => $request['state'],
            'city'              => $request['city'],
            'postal_code'       => $request['postal_code'],
            'address'           => $request['address'],
            'completed'         => 1,
        ]);
        alert()->flash('خوش آمدید', 'success', [
            'text' => "حساب کاربری شما با موفقیت تکمیل شد",
            'showConfirmButton' => false,
            'timer' => 2000
        ]);

        return redirect()->home();
    }

    protected function storeStepTwoUpdate(Request $request)
    {
        $this->validate($request,[
            'tell'              => ['required'],
            'urgent_phone_no'   => ['required'],
            'state'             => 'required|string|exists:provinces,name',
            'city'              => 'required|string|exists:cities,name',
            'postal_code'       => 'required|digits:10|numeric',
            'address'           => 'required|max:300'
        ]);
        $request = $request->all();
        $user = User::whereId(Auth::id())->first();
        $user->update([
            'tell'              => $request['tell'],
            'urgent_phone_no'   => $request['urgent_phone_no'],
            'state'             => $request['state'],
            'city'              => $request['city'],
            'postal_code'       => $request['postal_code'],
            'address'           => $request['address'],
        ]);

    }

    protected function storeStepTwoUpdate2(Request $request)
    {
        $this->validate($request,[
            'fname'             => ['required','max:60'],
            'lname'             => ['required','max:60'],
            'melli_code'        =>['required'],
            'tell'              => ['required'],
            'urgent_phone_no'   => ['required'],
            'state'             => 'required|string|exists:provinces,name',
            'city'              => 'required|string|exists:cities,name',
            'postal_code'       => 'required',
            'address'           => 'required|max:300'
        ]);
        $request = $request->all();
        $user = User::whereId(Auth::id())->first();
        $user->update([
            'fname'             => $request['fname'],
            'lname'             => $request['lname'],
            'melli_code'        => $request['melli_code'],
            'tell'              => $request['tell'],
            'urgent_phone_no'   => $request['urgent_phone_no'],
            'state'             => $request['state'],
            'city'              => $request['city'],
            'postal_code'       => $request['postal_code'],
            'address'           => $request['address'],
            'completed'         => 1,
        ]);

    }

    protected function activeShow(Request $request)
    {

        $request = $request->all();
        if (isset($request['refurl'])) {
            $user = User::where('token', $request['refurl'])->first();
            if ((URL::previous() == url('/register') ||URL::previous() == url('/register/activation') || URL::previous() == url('/activation') . '?refurl=' . $request['refurl']) && !is_null($request['refurl'])) {
                if (isset($user)) {
                    if ($user->token == $request['refurl'] && $user->confirmed != 1) {
                        return view('/public/SiteFront/auth/smsConfirm', [
                            'mobileNo' => $user->mobile,
                            'token' => $request['refurl'],
                        ]);
                    }
                } else {
                    return redirect()->back();
                }
            }
        }

        return redirect(route('home'));
    }

    protected function activationReq(Request $request)
    {
        $request = $request->all();
        $user = User::where('mobile', $request['mobile'])->first();
        if (isset($user)) {
            Event::fire(new RegisterEvent($user));
        }

    }

    protected function active(Request $request)
    {

        $this->validate($request, [
            'activationCode' => ['required', 'digits:5'],
        ]);


        $request = $request->all();
        $activationCode = $request['activationCode'];

        $user1 = User::where('token', $request['token'])->first();

        if (isset($user1)) {

            $mobile = $user1->mobile;
            $user = UserActivationSms::where('mobile', $mobile)->first();

            if (!empty($user)) {

                if ($user->activationCode == $activationCode) {

                    $activeUser = User::where('mobile', $user->mobile)->first();


                    $activeUser->confirmed = 1;
                    $activeUser->token = null;
                    $activeUser->save();
                    deleteSimple($user);

                    Auth::loginUsingId($activeUser->id);
                    Cart::addToCartInLogin();

                    alert()->flash('خوش آمدید', 'success', [
                        'text' => "حساب کاربری شما با موفقیت فعال گردید",
                        'showConfirmButton' => false,
                        'timer' => 2000
                    ]);

                    if (URL::previous() == url('/activation') . '?refurl=' . $request['token']) {
                        return redirect(route('completeQuestion'));
                    } else {
                        return redirect('/product/ss');
                    }
                } else {
                    alert()->flash('خطا!', 'error', [
                        'text' => "کد فعال سازی معتبر نمی باشد",
                        'type' => 'error',
                        'showConfirmButton' => true,
                        'confirmButtonText' => 'باشه',
                    ]);

                    return redirect('/activation/?refurl=' . $request['token']);
                }
            } else {

                $activeUser = User::whereraw('mobile', $mobile)->first();
                if (!empty($activeUser) && $activeUser->active == 1) {
                    alert()->flash('خطا!', 'error', [
                        'text' => "این حساب کاربری قبلا فعال گردیده است",
                        'type' => 'error',
                        'showConfirmButton' => true,
                        'confirmButtonText' => 'باشه',
                    ]);

                    return redirect('/activation/?refurl=' . $request['token']);
                } else {
                    alert()->flash('خطا!', 'error', [
                        'text' => "احتمالا کد فعالسازی ارسال نشده دوباره آن را درخواست کنید",
                        'type' => 'error',
                        'showConfirmButton' => true,
                        'confirmButtonText' => 'باشه',
                    ]);

                    return redirect('/activation/?refurl=' . $request['token']);
                }
            }
        } else {

            return redirect()->home();
        }
    }

    protected function noEdit(Request $request)
    {
        $this->validate($request, [
            'mobile' => ['required', 'digits:11', 'regex:/^(0)?(9[0-9]{9})$/', 'unique:users'],
        ]);
        $request = $request->all();

        $mobileOld = encrypt_decrypt('decrypt', $request['token']);

        $user = User::where('mobile', $mobileOld)->first();

        $userActive = UserActivationSms::where('mobile', $mobileOld)->first();

        if (isset($user)) {
            $user->update(['mobile' => $request['mobile']]);
            if (isset($userActive)) {
                deleteSimple($userActive);
                Event::fire(new RegisterEvent($user));
            }
        }

        alert()->flash('ویرایش شد', 'success', [
            'text' => "کد فعال سازی به شماره جدید ارسال شد",
            'showConfirmButton' => false,
            'timer' => 1800
        ]);

        return redirect()->back();
    }

    protected function completeQuestion()
    {

        $user = User::whereId(Auth::id())->first();

        if (isset($user)) {
            if ($user->completed == 0) {

                return view('public.SiteFront.auth.registerCompQue');
            }
        }
        return redirect()->home();
    }

    public function sendActivationCode()
    {
        return view('public.SiteFront.auth.registerActivatinReq');
    }

    public function activationCodeSend(Request $request)
    {
        $this->validate($request, [
            'mobile' => ['required', 'exists:users,mobile', 'digits:11', 'regex:/^(0)?(9[0-9]{9})$/'],
        ]);
        $user = User::whereMobile($request->mobile)->first();
        if (isset($user)) {
            if ($user->confirmed == 0) {
                Event::fire(new RegisterEvent($user));
                $user->update(['token'=> encrypt_decrypt('encrypt', $request->mobile)]);
                return redirect($this->redirectPath($user));
            } else {
                alert()->flash('توجه', 'info', [
                    'text' => "حساب کاربری قبلا فعال شده است. در صورت نیاز میتوانید از بخش فراموشی رمز، رمز خودرا مجددا بازیابی کنید",
                    'showConfirmButton' => true,
                ]);
                return back();
            }
        } else {
            alert()->flash('خطا', 'error', [
                'text' => "چنین کاربری وجود ندارد",
                'showConfirmButton' => true,
            ]);
            return back();
        }

    }
}
